	function [h, Err] = L_PHASE_LP_HB_FIR(Norder, wsT)
 
	%	Computes the impulse response of an even-order, half-band, equiripple,  
	%	linear-phase lowpass FIR filters using the fast algorithm described in [1].
	%	The passband edge is:  wcT =  - wsT
	%
	%   inputs:
	%		Norder: filter order, i.e., Norder+1 taps.  MUST BE EVEN
	% 		wcT: passband cutoff edge (0 < wcT < /2)
	%
	%	outputs:
	%		h 	is vector with the impulse response of an even-order half-band 
	%			linear-phase FIR filter, where N = 4m+2 and m is an integer.
	%	 	Err is the maximal error in the error function.	
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB
		
	% 	Modified by: 	LW 2005-06-10, 2016-03-05
	% 	Copyright:		by authors - not released for commercial use	 
	% 	Version:		1 
	% 	Known bugs:		
	% 	Report bugs to:	Wanhammar@gmail.com
	%
	% 	References:	[1]	Willson Jr. A.N. and Orchard H.J.: A Design Method for 
	%					Half-Band FIR Filters, IEEE Trans. on Circuits and Systems, 
	%					Part I, Vol. 45, No. 1, pp. 95-101, Jan. 1999.
	
	if (rem(Norder-2, 4) ~= 0), error('The filter order must be even and N = 4m + 2'), return; end
	if (wsT <= pi/2 | wsT <= 0), error('Incorrect passband edge'), return; end
	
	n1 = 0.5*Norder;		% Norder = 2n1 = even and 2n1+1 taps	
	n2 = n1-1;
	n4 =  2^floor(log2(n1));
	if (n4 < 8), n4 = 8; end
	n3 = n1 + n4;
	sign = (-1)^mod(floor(n2/2), 2);
	wcT = pi - wsT;
	alpha = sin(wcT);
	gamma = cos(wcT);
	
	% Compute the optimum number of sample points in n3
	rm = 10^6;
	GOTO = 1;	
	while GOTO < 4
		switch GOTO
		case 1
			n5 = floor(n3/2); 
			ne = mod(n3, 2);	% even/odd
			n6 = n5 + ne;
			bn = 1/n3;
			th = pi/(2*n3);
			GOTO = 2;
			if (n4 == 0), GOTO = 4; end
		case 2
			p = 0;
			q = 0;
			for m = 1:n5
				x = (2*m-1)*th;
				y = n1*x;
				g = bn/sqrt(1 - (alpha*cos(x))^2);
				p = p + g*sin(y)/sin(x);
				q = q + g*cos(y)/cos(x);
			end
			q = q*sign;
			if (ne == 1),
				g = bn*0.5;
				p = p + g*sign;
				q = q + g*n1;
			end
			r = abs((p*gamma-0.5)/(q-0.5)) - 1;
			if (abs(r) < rm),
				rm = abs(r);
				nt = n3;
			end
			n4 = floor(n4/2);
			if (r > 0),
				n3 = n3 + n4;
			else 
				n3  = n3 - n4;
			end
			GOTO = 3;
			if (n4 > 0),  GOTO = 1; end
		case 3
			n3 = nt;
			GOTO = 1;
		end	
	end

	% Compute a(i)
	f = zeros(1, n5);
	for m = 1:n5
		f(m) = bn/sqrt(1 - (alpha*cos((2*m-1)*th))^2);
	end
	if (ne == 1)
		f(n6) = bn*0.5;
	end
	a = zeros(1, n2+1);
	for k = 0:2:n2
		tk = k*th;
		a(k+1) = 0;
		for m = 1:n6
			a(k+1) = a(k+1) + f(m)*cos((2*m-1)*tk);
		end
	end
	a(1) = a(1)*0.5;
	
	% Remove scale factor on variable of Chebyshev polynomials
	d = zeros(n2+3, n2+2);
	b = zeros(1, n2+1);
	beta = 1/alpha;
	d(3, 1) = beta;
	d(2, 2) = 1;
	b(1) = a(1);
	for j = 1:n2
		p = a(j+1);
		for i = mod(j,2):2:j
			d(i+2, j+2) = beta*(d(i+1, j+1) + d(i+3, j+1)) - d(i+2, j);
			if (i == 1),
				d(3, j+2) = d(3, j+2) + beta*d(2, j+1);
			end
			if p ~= 0
				b(i+1) = b(i+1) + p*d(i+2, j+2);
			end
		end
	end
	for i = 2:4:n2
		b(i+1) = -b(i+1);
	end

	% Solve for h0(k) from b(k)
	b(1) = 2*b(1);
	b(n1+2) = 0;
	h0(1) = 0.5;
	for i = 1:2:n1
		h0(i+1) = 0.5*(b(i) + b(i+2));
	end
	p = -0.5;
	for j = 1:2:n1
		p = p + 2*h0(j+1);
	end
	Err = abs(p);
	Amin = -H_2_MAG(Err);
	%	disp(sprintf('Amin = %2.4f dB', Amin));	% If needed
	Amax = H_2_MAG((1 + Err)/(1 - Err));
	%	disp(sprintf('Amax = %2.8f dB', Amax));	% If needed

	% Determine the complete impulse response, h(n), from the single-sided h0(k)
	h = cat(2,fliplr(h0), h0(2:end));
